//
//  TodoItem.swift
//  Do It
//
//  Created by Jim Dovey on 8/23/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import Foundation
import SwiftUI

struct TodoItemList: Codable, Identifiable, Hashable {
    var id: UUID = UUID()
    var name: String
    var color: Color
    var icon: String
    
    var items: [UUID] = []

    enum Color: Hashable {
        case red
        case green
        case blue
        case orange
        case yellow
        case pink
        case purple
        case custom(hue: Double, saturation: Double, brightness: Double)
        
        static var allPredefinedColors: [Color] {
            [.red, .green, .blue, .orange, .yellow, .pink, .purple]
        }
        
        static func random() -> Color {
            allPredefinedColors.randomElement()!
        }
    }
}

struct TodoItem: Codable, Identifiable, Hashable {
    var id: UUID = UUID()
    var title: String
    var priority: Priority
    var notes: String?
    var date: Date?
    var listID: UUID
    var completed: Date?

    enum Priority: String, Codable, CaseIterable {
        case low
        case normal
        case high
        case urgent
    }
    
    var complete: Bool {
        get {
            guard let date = completed else { return false }
            return date < Date()
        }
        set {
            if newValue {
                completed = Date()
            }
            else {
                completed = nil
            }
        }
    }
}

extension TodoItem.Priority: Comparable {
    static func < (lhs: TodoItem.Priority, rhs: TodoItem.Priority) -> Bool {
        return allCases.firstIndex(of: lhs)! < allCases.firstIndex(of: rhs)!
    }
}

extension TodoItem.Priority {
    var localizedName: LocalizedStringKey {
        LocalizedStringKey(rawValue.localizedCapitalized)
    }
}

extension TodoItemList.Color {
    var uiColor: SwiftUI.Color {
        switch self {
        case .red: return Color.red
        case .green: return Color.green
        case .blue: return Color.blue
        case .orange: return Color.orange
        case .yellow: return Color.yellow
        case .pink: return Color.pink
        case .purple: return Color.purple
        case let .custom(h, s, b):
            return Color(hue: h, saturation: s, brightness: b)
        }
    }
}

extension TodoItemList.Color: Codable {
    init(from decoder: Decoder) throws {
        let container = try decoder.singleValueContainer()
        let str = try container.decode(String.self)
        switch str {
        case "red": self = .red
        case "green": self = .green
        case "blue": self = .blue
        case "orange": self = .orange
        case "yellow": self = .yellow
        case "pink": self = .pink
        case "purple": self = .purple

        case _ where str.hasPrefix("hsb:"):
            // An HSB color triple
            let colors = str.dropFirst(4)
                .components(separatedBy: ",")
                .compactMap(Double.init)
                .filter { (0.0...1.0).contains($0) }
            guard colors.count == 3 else {
                throw DecodingError.dataCorruptedError(
                    in: container,
                    debugDescription: "Invalid HSB color value \(str)")
            }
            self = .custom(hue: colors[0], saturation: colors[1], brightness: colors[2])

        default:
            throw DecodingError.dataCorruptedError(
                in: container,
                debugDescription: "Unrecognized color: \"\(str)\"")
        }
    }

    func encode(to encoder: Encoder) throws {
        var container = encoder.singleValueContainer()
        switch self {
        case .red: try container.encode("red")
        case .green: try container.encode("green")
        case .blue: try container.encode("blue")
        case .orange: try container.encode("orange")
        case .yellow: try container.encode("yellow")
        case .pink: try container.encode("pink")
        case .purple: try container.encode("purple")
        case let .custom(h, s, b):
            try container.encode("hsb:\(h),\(s),\(b)")
        }
    }
}
